// FesSelfMovement.js Ver.2.0.1
// MIT License (C) 2022 あわやまたな
// http://opensource.org/licenses/mit-license.php
﻿
/*:
* @target MV MZ
* @plugindesc ツクールフェスにあった単純な自律移動を再現します。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/485674614.html
* @help
* 自律移動　タイプ：固定　にし、下記のタグを入力することで新たな移動タイプを
* 使用可能になります。
*
*【オプション】
* イベントの設定したいページの１行目に注釈を置き、以下の形式で入力します。
* そうする事でページ切り替え時の初期状態を設定できます。
* ・プレイヤーから遠ざかる
* <fesSelfMove:遠ざかる>
* <fesSelfMove:moveAway>
* <fesSelfMove:1>
*
* ・横に往復
* <fesSelfMove:横に往復>
* <fesSelfMove:horz>
* <fesSelfMove:2>
*
* ・縦に往復
* <fesSelfMove:縦に往復>
* <fesSelfMove:vert>
* <fesSelfMove:3>
*
* 「遠ざかる」はプレイヤーから遠ざかる行動を自律移動仕様で実行します。
* 「横に往復」は直進→壁にぶつかる→左右反転→直進を繰り返します。
* 「縦に往復」は直進→壁にぶつかる→上下反転→直進を繰り返します。
*
* 【向き】
* 下：2　左：4　右：6　上：8
* 往復を開始する向きを設定します。
* 未入力だと左または下になります。
* （例）
* <fesSelfMove:横に往復,右>
* <fesSelfMove:横に往復,6>
* <fesSelfMove:horz,right>
* <fesSelfMove:horz,6>
* <fesSelfMove:2,6>
* <fesSelfMove:縦に往復,上>
* <fesSelfMove:縦に往復,8>
* <fesSelfMove:vert,up>
* <fesSelfMove:vert,8>
* <fesSelfMove:3,8>
*
* ［更新履歴］
* 2022/02/19：Ver.1.0.0　公開。
* 2022/02/20：Ver.1.1.0　往復のウェイトを補正。
* 2023/02/15：Ver.2.0.0　リニューアル。
* 2023/03/19：Ver.2.0.1　微修正。
*
* @param fesMode
* @text フェスモード
* @desc 往復中のキャラクターをイベントで挟むとカオスムーブになります。
* @type boolean
* @default false
*
* @param duration
* @text 反転時間
* @desc 往復で障害物に当たってから反転するまでにかかる時間。
* @type number
* @default 1
* @min 1
*
* @param simplifyRandom
* @text 自律移動：ランダムの簡素化
* @desc ランダムの偏りをなくします。
* @type boolean
* @default false
*
* @param simplifyApproach
* @text 自律移動：近づくの簡素化
* @desc 挙動の不確定要素を取り除きます。
* @type boolean
* @default false
*
* @param simplifyEscape
* @text 自律移動：逃げるの簡素化
* @desc 挙動の不確定要素を取り除きます。
* @type boolean
* @default false
*
*/
﻿
'use strict';
{
	const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
	const param = PluginManager.parameters(pluginName);
	const fesMode = param['fesMode'] === 'true';
	const duration = Number(param['duration']);
	const simplifyRandom = param['simplifyRandom'] === 'true';
	const simplifyApproach = param['simplifyApproach'] === 'true';
	const simplifyEscape = param['simplifyEscape'] === 'true';
﻿
	const _Game_Event_initMembers = Game_Event.prototype.initMembers;
	Game_Event.prototype.initMembers = function() {
		this._fesSelfMoveType = 0;
		this._fesSelfMoveDir = 0;
		_Game_Event_initMembers.call(this);
	};
	
	const TYPE_MOVE_AWAY = 1;
	const TYPE_RECIPROCATE_HORIZONTALLY = 2;
	const TYPE_RECIPROCATE_VERTICALLY = 3;
﻿
	const _Game_Event_updateSelfMovement = Game_Event.prototype.updateSelfMovement;
	Game_Event.prototype.updateSelfMovement = function() {
		if (this._moveType > 0) {
			_Game_Event_updateSelfMovement.call(this);
		} else {
			this.updateFesSelfMovement();
		}
	};
﻿
	Game_Event.prototype.updateFesSelfMovement = function() {
		if (!this._locked && this.isNearTheScreen() && this.checkStop(this.stopCountThreshold())) {
			switch (this._fesSelfMoveType) {
				case TYPE_MOVE_AWAY:
					this.moveTypeAwayFromPlayer();
					break;
				case TYPE_RECIPROCATE_HORIZONTALLY:
					this.moveTypeFesRecipHorz();
					break;
				case TYPE_RECIPROCATE_VERTICALLY:
					this.moveTypeFesRecipVert();
					break;
				default:
			}
		}
	};
﻿
	Game_Event.prototype.moveTypeAwayFromPlayer = function() {
		if (this.isNearThePlayer()) {
			switch (Math.randomInt(6)) {
				case 0:
				case 1:
				case 2:
				case 3:
					this.moveAwayFromPlayer();
					break;
				case 4:
					this.moveRandom();
					break;
				case 5:
					this.moveForward();
					break;
			}
		} else {
			this.moveRandom();
		}
	};
﻿
	Game_Event.prototype.moveTypeFesRecipHorz = function() {
		let lastDir = this._fesSelfMoveDir;
		if (lastDir !== 6) {
			lastDir = 4;
			this._fesSelfMoveDir = lastDir;
		}
		this.moveTypeFesRecip(lastDir);
	};
﻿
	Game_Event.prototype.moveTypeFesRecipVert = function() {
		let lastDir = this._fesSelfMoveDir;
		if (lastDir !== 8) {
			lastDir = 2;
			this._fesSelfMoveDir = lastDir;
		}
		this.moveTypeFesRecip(lastDir);
	};
﻿
	Game_Event.prototype.moveTypeFesRecip = function(d) {
		this.moveStraight(d);
		if (!this.isMovementSucceeded()) {
			this._fesSelfMoveDir = this.reverseDir(d);
			
		}
	};
﻿
	Game_Event.prototype.moveTypeFesRecip = function(d) {
		if (this.canPass(this._x, this._y, d)) {
			this.moveStraight(d);
		} else {
			const rd = this.reverseDir(d);
			if ((fesMode || !this.canReverseFesRecip(d)) && this.canReverseFesRecip(rd)) {
				this.moveStraight(rd);
			}
			this._fesSelfMoveDir = rd;
			this._stopCount = this.stopCountThreshold() - (duration - 1);
		}
	};
﻿
	Game_CharacterBase.prototype.canReverseFesRecip = function(d) {
		const x2 = $gameMap.roundXWithDirection(this._x, d);
		const y2 = $gameMap.roundYWithDirection(this._y, d);
		if (this.isCollidedWithCharacters(x2, y2)) {
			return true;
		}
		return false;
	};
﻿
	if (simplifyRandom) {
		Game_Event.prototype.moveTypeRandom = function() {
			this.moveRandom();
		};
	}
﻿
	if (simplifyApproach) {
		Game_Event.prototype.moveTypeTowardPlayer = function() {
			if (this.isNearThePlayer()) {
				this.moveTowardPlayer();
			} else {
				this.moveRandom();
			}
		};
	}
﻿
	if (simplifyEscape) {
		Game_Event.prototype.moveTypeAwayFromPlayer = function() {
			if (this.isNearThePlayer()) {
				this.moveAwayFromPlayer();
			} else {
				this.moveRandom();
			}
		};
	}
﻿
	const _Game_Event_clearPageSettings = Game_Event.prototype.clearPageSettings;
	Game_Event.prototype.clearPageSettings = function() {
		_Game_Event_clearPageSettings.call(this);
		this._fesSelfMoveType = 0;
		this._fesSelfMoveDir = 0;
	};
﻿
	const _Game_Event_setupPageSettings = Game_Event.prototype.setupPageSettings;
	Game_Event.prototype.setupPageSettings = function() {
		_Game_Event_setupPageSettings.call(this);
		this.setupPageSettingsFesSelfMovement();
	};
﻿
	Game_Event.prototype.setupPageSettingsFesSelfMovement = function() {
		const page = this.page();
		this._fesSelfMoveType = page.fesSelfMoveType;
		this._fesSelfMoveDir = page.fesSelfMoveDir;
	};
﻿
	const _DataManager_onLoad = DataManager.onLoad;
	DataManager.onLoad = function(object) {
		_DataManager_onLoad.call(this, object);
		if (!!(object.data && object.events) && Array.isArray(object.events)) {
			for (const event of object.events) {
				if (event && event.pages){
					extractMetadata(event);
				}
			}
		}
	};
﻿
	function extractMetadata(data) {
		for (const page of data.pages) {
			const comment = findComment(page);
			const commentData = {"note": comment};
			DataManager.extractMetadata(commentData);
			addPageSettings(page, commentData.meta);
		}
	}
﻿
	function findComment(page) {
		const list = page.list;
		if (!list[0] && list[0].code !== 108) {
			return "";
		}
		let comment = list[0].parameters[0];
		for (let i = 1; list[i] && list[i].code === 408; i++) {
			comment += list[i].parameters[0];
		}
		return comment;
	}
﻿
	function addPageSettings(page, meta) {
		const data = (meta['fesSelfMove'] || "").split(",");
		let moveType = 0;
		let dir = 0;
		switch (data[0]) {
			case "moveAway":
			case "遠ざかる":
				moveType = TYPE_MOVE_AWAY;
				break;
			case "horz":
			case "横に往復":
				moveType = TYPE_RECIPROCATE_HORIZONTALLY;
				break;
			case "vert":
			case "縦に往復":
				moveType = TYPE_RECIPROCATE_VERTICALLY;
				break;
		}
		switch (data[1]) {
			case "down":
			case "下":
				dir = 2;
				break;
			case "left":
			case "左":
				dir = 4;
				break;
			case "right":
			case "右":
				dir = 6;
				break;
			case "up":
			case "上":
				dir = 8;
				break;
		}
		page.fesSelfMoveType = Number(moveType || 0);
		page.fesSelfMoveDir = Number(dir || 0);
	}
}
